module cmdfit_subs

  use colteff_subs
  use red_ext_interp
  use cmdfit_system
  use mass_functions

  implicit none

  real, private :: extra_mag, high_tau=-1.0

  contains

  real function high_calc_tau()

    high_calc_tau=high_tau/2.0

  end function high_calc_tau

  real function sum_tau2(tau2_grid, axdata, axisnam, min1, max1, min2, max2)

    ! This sums the probability from a given set of tau2 pixels.
    ! The only clever bit is that if it finds one of the axes is a
    ! log axis it will weight the pixels so that the sum is done
    ! as though it were a linear axis.

    real, dimension(:,:), intent(in) :: tau2_grid, axdata
    character(len=*), dimension(:), intent(in) :: axisnam
    ! The region to be summed.  Clumsy to have to put it in, but otherwise
    ! you have to do clumsy things with subsetting axdata.
    integer, intent(in) :: min1, max1, min2, max2

    ! Locals.
    integer :: iaxis1, iaxis2
    real, parameter :: big=2.0*log(huge(tau2_grid))
    ! Set these up to be the size of the pixels.
    real, dimension(size(tau2_grid,1)) :: delta1
    real, dimension(size(tau2_grid,2)) :: delta2

    delta1=1.0
    if (axisnam(1)(1:3) == 'LOG') delta1(1:size(delta1))=10.0**axdata(1:size(tau2_grid,1),1)
    delta2=1.0
    if (axisnam(2)(1:3) == 'LOG') delta2(1:size(delta2))=10.0**axdata(1:size(tau2_grid,2),2)

    sum_tau2=0.0
    do iaxis2=min2, max2
      do iaxis1=min1, max1
        if (tau2_grid(iaxis1, iaxis2) < big/2.0) &
        sum_tau2=sum_tau2 + &
        (delta1(iaxis1)*delta2(iaxis2)*exp(-0.5*(tau2_grid(iaxis1,iaxis2))))
      end do
    end do

  end function sum_tau2

  subroutine natural_norm(mag_max, mag_min, data, axdata)

    ! This subroutine normalises the model, so that it integrates to
    ! one between the faintest and brightest stars.

    ! Bug corrected in 2012 where it dod not multiply by the pixel area.
    
    real, intent(in) :: mag_max, mag_min
    real, intent(inout), dimension(:,:) :: data, axdata

    real :: pix_area
    integer :: imag_low, imag_high
    integer, dimension(2) :: naxis

    ! Fnd the area of a pixel.
    naxis(1)=size(data,1)
    naxis(2)=size(data,2)
    pix_area = (abs(axdata(naxis(1),1)-axdata(1,1))/real(naxis(1)-1)) &
             * (abs(axdata(naxis(2),2)-axdata(1,2))/real(naxis(2)-1))
        
    ! Find the nearest pixel position in mag (avoiding problems if
    ! magnitude axes run in the wrong direction) for the brightest
    ! and faintest stars.
    imag_low =min( minloc(abs( mag_min-axdata(1:naxis(2),2) ),1), &
                   minloc(abs( mag_max-axdata(1:naxis(2),2) ),1))
    imag_high=max( minloc(abs( mag_min-axdata(1:naxis(2),2) ),1), &
                   minloc(abs( mag_max-axdata(1:naxis(2),2) ),1))
    data=data/(pix_area*sum(data(:,imag_low:imag_high)))


  end subroutine natural_norm

  subroutine make_star(age, iso_file, bc_file, ext_file, colnam, mass_function, low_mass, up_mass, &
  pwr_mass, bin_frac, red, mag, col, p_mass, &
  nstars, flag_out, p_teff, p_logg, p_lbol)

    ! The age required.
    real, intent(in) :: age
    ! The files to use for the interiors and bolometric corrections.
    character(len=*), intent(in) :: iso_file, bc_file, ext_file
    ! The names of the magnitude and/or colours to be created.
    character(len=10), dimension(2), intent(inout) :: colnam
    ! The form of the mass function to be used.
    character(len=*) :: mass_function
    ! If the form is "power_law" then you need to suppy maximum and
    ! minumum masses.  The function power_law_mass describes pwr_mass.
    ! If the form is "kroupa"  then pwr_mass is ignored.
    ! If the form is "salpeter" then -2.35 is used.
    real, intent(in) :: up_mass, low_mass, pwr_mass 
    ! The fraction of systems which are binaries.
    real, intent(in) :: bin_frac
    ! If the Bell extinction tables are there, the nominal E(B-V), if not the
    ! reddening. 
    real :: red
    ! The colour and magnitude of the stars created. 
    real, intent(out) :: mag, col
    ! The mass of the primary star.
    real, intent(out) :: p_mass
    ! The number of stars in the system.
    integer, intent(out) :: nstars
    ! A character flag, 'OK' if things are.
    character(len=*), intent(out) :: flag_out
    ! The effective temperature and gravity of the primary star.
    ! Calculating these costs time, so if the caller does not need them
    ! it faster to omit them.
    real, optional, intent(out) :: p_teff, p_logg, p_lbol

    ! Locals.
    logical, save :: first=.true.
    character(len=50) :: isofil
    real, save :: col1, col2, mag1, mag2, ebmv, av
    real :: harvest, grad
    integer, parameter :: max_multiplicity=2
    real, dimension(max_multiplicity) :: mass, uflux, bflux, vflux, iflux
    real :: umb, bmv, vmi, v, logg, teff, test
    integer :: iminus
    logical :: v_vmi_like, v_bmv_like, umb_bmv_like
    character(len=20) :: flag, work_flag
    logical :: extrap

    type(chrone), save :: iso
    integer :: icol1_1, icol2_1, icol1_2, icol2_2
    logical :: ok
    character(len=50), dimension(:), allocatable :: workflag

    real :: work, red_in1, red_in2

    real, save :: last_age=0.0

    if (abs(age-last_age) > 2.0*tiny(age)) then
      ! Create an isochrone.       
      iso%iso_file=iso_file
      iso%bc_file=bc_file
      iso%age=age
      call iso_calc(iso)
      last_age=age
    end if

    ! First find out if the colours are of the form V vs V-I or V vs B-V.
    umb_bmv_like=.false.
    v_vmi_like=.false.
    v_bmv_like=.false.
    ! First is this colour/colour?
    iminus=index(colnam(1), '-') 
    if (iminus > 0) then
      ! O.K., is it in the correct order?
      if (trim(colnam(1)(iminus+1:len(colnam(1)))) /= &
      colnam(2)(1:index(colnam(2), '-')-1)) then
        print*, 'Colour ', colnam(1), ' not of form U-B vs B-V.'
        stop
      else
        umb_bmv_like=.true.
      end if
    else
      iminus=index(colnam(2), '-') 
      if (iminus == 0) then
        print*, 'Cannot find the minus in ', colnam(2)
        stop
      else
        if (trim(colnam(1)) == colnam(2)(1:iminus-1)) then
          v_vmi_like=.true.
        else if (trim(colnam(1)) == &
        trim(colnam(2)(iminus+1:))) then
          v_bmv_like=.true.
        else
          print*, 'Cannot find ', trim(colnam(1)), ' in ', &
          trim(colnam(2))
          stop
        end if
      end if
    end if

    ok=find_bc_combination(colnam(1), iso%colnam, iso%ncol, icol1_1, icol2_1)
    !if (first) then
    !  print*, ok, icol1_1, icol2_1, iso%colnam(icol1_1)
    !  if (icol2_1 > 0) print*, iso%colnam(icol2_1)
    !end if
    ok=find_bc_combination(colnam(2), iso%colnam, iso%ncol, icol1_2, icol2_2)
    !if (first) then
    !  print*, ok, icol1_2, icol2_2, iso%colnam(icol1_2), iso%colnam(icol2_2)
    !  print*, umb_bmv_like, v_vmi_like, v_bmv_like
    !  first=.false.
    !end if

    nstars=0
    each_component: do

      if (nstars == max_multiplicity) exit each_component

      if (nstars == 0) then 
        nstars=nstars+1
        if (trim(mass_function) == 'power_law') then
          mass(1)=power_law_mass(low_mass, up_mass, pwr_mass)
        else if (trim(mass_function) == 'salpeter') then
          mass(1)=power_law_mass(low_mass, up_mass, -2.35)
        else if (trim(mass_function) == 'kroupa') then
          mass(1)=kroupa_star_mass(low_mass, up_mass)
        else
          print*, 'I do not have mass function form for ', trim(mass_function)
          stop
        end if
        ! Work out the primary's temperature.
        if (present(p_teff)) p_teff = &
        10.0**linintflg(iso%mass, iso%teff, mass(nstars), flag, iso%massflg, iso%teffflg)
        ! And gravity.
        if (present(p_logg)) p_logg = &
        linintflg(iso%mass, iso%logg, mass(nstars), flag, iso%massflg, iso%loggflg)
        if (present(p_lbol)) p_lbol = &
        linintflg(iso%mass, iso%lbol, mass(nstars), flag, iso%massflg, iso%lbolflg)
      else
        call random_number(harvest)
        if (mass(1) < 14.4) then 
          ! First does this star have a companion?  The orginal 
          ! condition (used in the work for the IAU general assembly
          ! paper) was teff < 30,000K, but this condition is better.
          if (harvest > bin_frac) exit each_component
          call random_number(harvest)
        else
          ! For O-stars the binary fraction is probably 67 percent, but
          ! let's be really pessimistic and say 75 percent, assuming
          ! a basic binary fraction of 0.5.
          if (harvest > 1.5*bin_frac) exit each_component
          ! Now the mass ratio.  There is an argument for a mass spike 
          ! that contains up to 20 percent of binaries at q=0.95 to 1.  
          ! Create this spike by losing all the low-q binaries.
          call random_number(harvest)
          if (harvest < 0.2) harvest = 1.0 - harvest/4.0
        end if
        nstars=nstars+1
        ! Now work out the mass of the companion.
        mass(nstars) = mass(nstars-1)*harvest 
        ! Hack to cut off at 0.25Mo for simulation of effect of
        ! binary wedge for original tau^2 paper.
        !if (mass(nstars) < 0.25) then
        !  flux1(nstars)=0.0
        !  flux2(nstars)=0.0
        !  cycle each_component
        !end if
      end if

      ! print*, 'Finding magnitude.'
      if (v_vmi_like .or. v_bmv_like) then
        v=linintflg(iso%mass, iso%pnt%col(icol1_1)%data, mass(nstars), flag, &
        iso%massflg, iso%pnt%col(icol1_1)%flag)
        if (trim(flag) == '1') flag='outside mass range'
      else if (umb_bmv_like) then
        ! In the colour-colour case get V from another isochrone.
        v=linintflg(iso%mass, iso%pnt%col(icol2_2)%data, mass(nstars), flag, &
        iso%massflg, iso%pnt%col(icol2_2)%flag)
        if (trim(flag) == '1') flag='outside mass range'
      end if
      ! Now, if the flag is not OK, continue, but note the fact.
      ! print*, 'Finding colour.'
      work_flag=flag
      if (v_bmv_like .or. umb_bmv_like) then
        if (icol2_2 == 0) then
          ! The files supplied B-V.
          bmv=linintflg(iso%mass, iso%pnt%col(icol1_2)%data, mass(nstars), flag, &
          iso%massflg, iso%pnt%col(icol1_2)%flag)
        else
          allocate(workflag(iso%npts))
          workflag=iso%pnt%col(icol1_2)%flag
          where(iso%pnt%col(icol2_2)%flag /= 'OK') workflag=iso%pnt%col(icol2_2)%flag
          bmv=linintflg(iso%mass, iso%pnt%col(icol1_2)%data-iso%pnt%col(icol2_2)%data, &
          mass(nstars), flag, iso%massflg, workflag)
          deallocate(workflag)
        end if
        if (trim(flag) == '1') flag='outside mass range'
        vmi=0.0
      else if (v_vmi_like) then
        if (icol2_2 == 0) then
          ! The files supplied V-I.
          vmi=linintflg(iso%mass, iso%pnt%col(icol1_2)%data, mass(nstars), flag, &
          iso%massflg, iso%pnt%col(icol1_2)%flag)
        else
          allocate(workflag(iso%npts))
          workflag=iso%pnt%col(icol1_2)%flag
          where(iso%pnt%col(icol2_2)%flag /= 'OK') workflag=iso%pnt%col(icol2_2)%flag
          vmi=linintflg(iso%mass, iso%pnt%col(icol1_2)%data-iso%pnt%col(icol2_2)%data, &
          mass(nstars), flag, iso%massflg, workflag)
          deallocate(workflag)
        end if
        if (trim(flag) == '1') flag='outside mass range'
        bmv=0.0
      end if
      if (flag /= 'OK') work_flag=flag
      if (umb_bmv_like) then
        ! In the colour-colour case U-B is stored as a magnitude.
        if (icol2_1 == 0) then
          ! The files supplied U-B.
          umb=linintflg(iso%mass, iso%pnt%col(icol1_1)%data, mass(nstars), flag, iso%massflg)
        else
          allocate(workflag(iso%npts))
          workflag=iso%pnt%col(icol1_1)%flag
          where(iso%pnt%col(icol2_1)%flag /= 'OK') workflag=iso%pnt%col(icol2_1)%flag
          umb=linintflg(iso%mass, iso%pnt%col(icol1_1)%data-iso%pnt%col(icol2_1)%data, &
          mass(nstars), flag, iso%massflg, workflag)
          deallocate(workflag)
        end if
        if (trim(flag) == '1') flag='outside mass range'
        if (flag /= 'OK') work_flag=flag
      else
        umb=0.0
      end if
      if (nstars == 1) then
        flag_out=work_flag
      else
        ! This is a secondary star.  
        if (trim(work_flag) /= 'OK') then
          ! Only possible error is that its dropped 
          ! off the bottom of the isochrone.  
          ! Set fluxes already to zero.
          uflux(nstars)=0.0
          bflux(nstars)=0.0
          vflux(nstars)=0.0
          iflux(nstars)=0.0
          cycle each_component
        end if
      end if
      ! And the gravity.
      ! print*, 'Finding gravity.'
      logg=linintflg(iso%mass, iso%logg, mass(nstars), flag, iso%massflg, iso%loggflg)
      !logg=isochr('mas', mass(nstars), 'log', age, iso_file, bc_file, colnam, flag)
      teff=10.0**linintflg(iso%mass, iso%teff, mass(nstars), flag, iso%massflg, iso%teffflg)
      !teff=10.0**isochr('mas', mass(nstars), 'tef', age, iso_file, bc_file, colnam, flag)
      ! For the reddening.
      if (abs(red) > tiny(red)) then
        !print*, 'About to do the reddening.', red, ext_file
        if (len_trim(ext_file) > 0) then
          if (v_bmv_like) then
            red_in1=bmv
            red_in2=v
          else if (v_vmi_like) then
            red_in1=vmi
            red_in2=v
          else if (umb_bmv_like) then
            red_in1=bmv
            red_in2=umb
          else
            print*, 'Hmm, that was unexpected.'
            stop
          end if
          if (bell_there(ext_file)) then
            call bell_ext(ext_file, colnam, red, logg, teff, red_in1, red_in2, extrap)
          else
            call reddening_point(colnam, ext_file, 0.0, red, red_in1, red_in2)
            extrap=.false.
          end if
          if (v_bmv_like) then
            bmv=red_in1
            v=red_in2
          else if (v_vmi_like) then
            vmi=red_in1
            v=red_in2
          else if (umb_bmv_like) then
            bmv=red_in1
            umb=red_in2
          end if
        else
          print*, 'No extinction file specified.'
          stop
        end if
        if (extrap) then
          if (trim(flag_out) == 'OK') flag_out='Extinction'
        end if
      end if
      uflux(nstars)=10.0**(-0.4*(v+bmv+umb))
      bflux(nstars)=10.0**(-0.4*(v+bmv))
      vflux(nstars)=10.0**(-0.4*v)
      iflux(nstars)=10.0**(-0.4*(v-vmi))

    end do each_component

    ! Add up all the components.
    if (v_vmi_like) then
      mag=-2.5*log10(sum(vflux(1:nstars)))
      col=-2.5*log10(sum(vflux(1:nstars))/sum(iflux(1:nstars)))    
    else if (v_bmv_like) then
      mag=-2.5*log10(sum(vflux(1:nstars)))
      col=-2.5*log10(sum(bflux(1:nstars))/sum(vflux(1:nstars)))
    else if (umb_bmv_like) then
      mag=-2.5*log10(sum(uflux(1:nstars))/sum(bflux(1:nstars)))
      col=-2.5*log10(sum(bflux(1:nstars))/sum(vflux(1:nstars)))
      extra_mag=-2.5*log10(sum(vflux(1:nstars)))
    end if
    p_mass=mass(1)


  end subroutine make_star

  real function v_for_ubv()

    v_for_ubv=extra_mag

  end function v_for_ubv

  subroutine iso_image_name(colnam, age, filnam)

    character(len=*), dimension(2), intent(in) :: colnam
    real :: age
    character(len=*), intent(out) :: filnam

    character(len=6) :: age_name

    write(age_name, 30) int(age+tiny(age)), nint(100*(age-int(age+tiny(age))))
30  format(i2.2, '.', i2.2, '0')

    filnam=trim(colnam(1)) //'_'// &
           trim(colnam(2)) //'_'// &
                age_name          //'.fit'

  end subroutine iso_image_name

  real function likelihood(data, axdata, grad, icol, imag, sig_col, &
    sig_mag, correlated, flag, sum_gauss)
    
    ! The unsmoothed image and axes.
    real, dimension(:,:), intent(in) :: data, axdata
    real, dimension(:), allocatable :: grad
    ! The co-ordinates of the position being considered.
    integer, intent(in) :: icol, imag
    real, intent(in) :: sig_col, sig_mag
    ! If correlated is zero the colour and magnitude uncertainties are taken
    ! to be uncorrelated.  If its 1, they are correlated as would
    ! be expected in I vs R-I; if its -1 as for V vs V-I (i.e. its whether 
    ! you add or subtract the colour to make the other magnitude).  
    ! If the uncertainties are correlated, make sure sig_col > sig_mag.
    integer, intent(in) :: correlated
    character, intent(out) :: flag

    real, optional, intent(out) :: sum_gauss

    ! We cut off the Gaussian smoothing the 2D distribution at some sigma.
    real, parameter :: sig_cut=3.0
    real, parameter :: pi=3.141592654

    ! Gradient in mag-mag space, and uncertainty in second magnitude.
    real :: mag_grad, mag_sig
    integer :: iicol, iimag, i1, i2
    real :: col, mag
    real :: work
    real, save :: area, pix1, pix2
    real :: d_col, d_mag, gauss, rho, gauss_norm
    logical, save :: first=.true.
    
    if (first) then
      ! Find the area of a pixel.
      pix1=(axdata(size(data,1),1)-axdata(1,1))/real(size(data,1)-1)
      pix2=(axdata(size(data,2),2)-axdata(1,2))/real(size(data,2)-1)
      area=abs(pix1*pix2)
      first=.false.
    end if

    ! Find the nearest positions in X and Y.
    col=axdata(icol,1)
    mag=axdata(imag,2)

    ! How far out do we need to go? 
    iicol=nint(sig_cut*sig_col/abs(pix1))
    iimag=nint(sig_cut*sig_mag/abs(pix2))

    ! For correlated uncertainties calculate mag_sig.
    if (correlated == 0) then
      mag_sig=sig_col
    else
      if (sig_mag > sig_col) then
        mag_sig=sig_mag/2.0
      else
        mag_sig=max(sqrt(sig_col**2.0 - sig_mag**2.0), sig_mag/2.0)
      end if
    end if

    ! Now we normalise the model so it is one within the magnitude
    ! range.


    if (present(sum_gauss)) then
      sum_gauss=0.0
      
      fudge: do i1=icol-iicol, icol+iicol

        ! This sums up the volume under the uncertainty ellipse which
        ! lies on the area of the model.  If it falls outside the colour
        ! range of the image we still add up the volume.  This is because
        ! monte trims in colour to save space.  In magnitude its a
        ! different story.  The model may well be non-zero outside the
        ! magnitude range, so we need to calculate what fraction of this 
        ! error ellipse falls within the monte grid.

        ! This works for all values of i1, even if they lie outside the grid.
        work=axdata(1,1) + (pix1*real(i1-1)) - col

        do i2=max(imag-iimag, 1), min(imag+iimag, size(data,2))
          
          d_mag=(axdata(i2,2)-mag)/sig_mag
          if (correlated == 0) then
            d_col=work/sig_col
          else if (correlated == 1) then
            d_col=(axdata(i2,2)-mag)+work
            d_col=d_col/mag_sig
          else if (correlated == -1) then
            d_col=(axdata(i2,2)-mag)-work
            d_col=d_col/mag_sig
          else 
            print*, 'Unacceptable value of ', correlated
            stop
          end if
          gauss=exp(-0.5*(d_mag**2.0 + d_col**2.0))
          sum_gauss=sum_gauss+(area*gauss)
          
        end do
      end do fudge

      sum_gauss=sum_gauss/(2.0*pi*mag_sig*sig_mag)

    end if


    ! Removed this if.  It used to check you didn't skip off the edge of the
    ! image.  Now the assmumption is that if you do, the values of the image
    ! pixels should be zero.
    !if (icol-iicol>0       .and. imag-iimag>0 .and. &
    !  icol+iicol<=size(data,1) .and. imag+iimag<=size(data,2)) then

      work=0.0

    
      outer: do i1=max(icol-iicol, 1), min(icol+iicol, size(data,1))

        if (allocated(grad)) then
          if (grad(i1) < tiny(grad(i1))) cycle outer
          ! I.e. if there are any non-zero data points at this colour.
          ! Calculate the normalisation.
          if (correlated == 0) then
            rho=(1.0/sig_mag)**2.0 + 1.0/(sig_col*grad(i1))**2.0
          else 
            mag_grad=1.0/(1.0-(1.0/grad(i1)))
            ! Avoid a numberical error if the two uncertainties are equal.
            mag_sig=max(sqrt(sig_col**2.0 - sig_mag**2.0), sig_mag/2.0)
            ! write(22,*) 'Got ', grad(i1), mag_grad, i1, mag_sig
            rho=(1.0/sig_mag)**2.0 + 1.0/(mag_sig*mag_grad)**2.0
            ! write(22,*) 'Which gave ', rho
          end if
          ! rho=sqrt(rho/pi)
          ! Correct the factor two error.
          rho=sqrt(0.5*rho/pi)
          gauss_norm=1.0
        else
          rho=1.0
          gauss_norm=1.0/(2.0*pi*mag_sig*sig_mag)
        end if



        do i2=max(imag-iimag, 1), min(imag+iimag, size(data,2))
          if (data(i1,i2) >= tiny(data(i1,i2))) then
            ! Find the colour difference from this point to the
            ! data point, and put it in terms of sigma.
            if (correlated == 0) then
              d_col=(axdata(i1,1)-col)/sig_col
            else if (correlated == 1) then
              ! But now comes the swindle to account for correlated
              ! uncertainties.  The colour distance we calculate, is 
              ! actually the distance in the other magnitude.
              d_col=(axdata(i2,2)-mag)+(axdata(i1,1)-col)
              ! Thus we have to divide by the uncertainty in the other 
              ! magnitude, reconstructed from the given uncertainties.
              d_col=d_col/mag_sig
            else if (correlated == -1) then
              d_col=(axdata(i2,2)-mag)-(axdata(i1,1)-col)
              d_col=d_col/mag_sig
            else 
              print*, 'Unacceptable value of ', correlated
              stop
            end if
            ! The same for the magnitude.  
            d_mag=(axdata(i2,2)-mag)/sig_mag
            ! gauss=exp(-1.0*(d_mag**2.0 + d_col**2.0))
            ! Correct the factor two error.
            gauss=exp(-0.5*(d_mag**2.0 + d_col**2.0))
            ! Now multiply the Gaussian at this point by the value of the
            ! model, the area of the pixel, and apply the normalisation.
            ! write(22,*) work, rho, area, gauss, data(i1,i2), i1, i2
            work=work+rho*area*gauss_norm*gauss*data(i1,i2)
          end if
        end do
      end do outer

      likelihood=work
      flag = 'O'

    !else

    !  likelihood=0.0
    !  flag = 'A'

    !end if

  end function likelihood


  real function func2d(a_par, dat, colnam, correlated, &
    distrib, distrib_flag, rnpts)

    use ark_file_io
    use define_star

    real, intent(in), dimension(:) :: a_par
    type(a_star), dimension(:), intent(in) :: dat
    character(len=*), dimension(2), intent(inout) :: colnam
    ! The correlation flag for s/r likelihood.
    integer, intent(in) :: correlated
    real, intent(out), dimension(:) :: distrib
    character, intent(out), dimension(:) :: distrib_flag
    real, optional, intent(out) :: rnpts

    integer :: i
    real, save :: last_age=-1
    real, dimension(:,:), allocatable, save :: data, axdata
    integer, dimension(2), save :: naxis
    real, dimension(:), allocatable, save :: grad, axgrad
    integer, save :: n_grad
    character(len=80), save :: ext_file
    integer :: iflag

    integer :: ipos, icol, imag
    real :: mag, col
    character(len=50) :: ifname

    real :: sum_gauss

    ! The nomrmalisation for the model.
    integer :: npix
    real :: mag_min, mag_max

    ! In this example case the function is simply a straight line with 
    ! a width given by the uncertainty in the second co-ordinate.

    if (abs(last_age-a_par(1)) > tiny(last_age)) then
      call iso_image_name(colnam, a_par(1), ifname)
      call nxtark_in(ifname)
      iflag=inpark(naxis, data, axdata)
      if (iflag < 0) then
        print*, 'Error reading 2D isochrone file ', ifname
        stop
      end if
      iflag=get_header_s('EXT_FILE', ext_file)
      if (iflag < 0) then
        print*, 'Error reading extinction file name from file ', ifname
        stop
      end if
      ! And add the directory name.
      ext_file=trim(data_dir())//ext_file
      last_age=a_par(1)
    end if

    ! Find the range of magnitudes in the data.
    find_range: do i=1, size(dat,1)
      ! Change to reddening-free absolute magnitude.
      mag=dat(i)%col(1)%data
      col=dat(i)%col(2)%data
      call reddening(colnam, ext_file, -a_par(2), -a_par(3), col, mag)
      if (i ==1) then
        mag_min=mag
        mag_max=mag
      else
        mag_min=min(mag_min, mag)
        mag_max=max(mag_max, mag)
      end if
    end do find_range
    call natural_norm(mag_max, mag_min, data, axdata)

    func2d = 0.0
    if (present(rnpts)) rnpts=0.0


    each_point: do i=1, size(dat,1)

      ! Change to reddening-free absolute magnitude.
      mag=dat(i)%col(1)%data
      col=dat(i)%col(2)%data
      call reddening(colnam, ext_file, -a_par(2), -a_par(3), col, mag)

      ! Find the nearest positions in X and Y.  (S/R likelihood will spot
      ! if we are close to the array edge).
      !icol=minval(minloc(abs(col-axdata(1:size(data,1),1))))
      icol=locate_nearest(axdata(1:size(data,1),1), col)
      !imag=minval(minloc(abs(mag-axdata(1:size(data,2),2))))
      imag=locate_nearest(axdata(1:size(data,2),2), mag)

      if (present(rnpts)) then
        distrib(i)=likelihood(data, axdata, grad, icol, imag, &
        dat(i)%col(2)%err, dat(i)%col(1)%err, correlated, distrib_flag(i), &
        sum_gauss)
        rnpts=rnpts+sum_gauss
      else
        distrib(i)=likelihood(data, axdata, grad, icol, imag, &
        dat(i)%col(2)%err, dat(i)%col(1)%err, correlated, distrib_flag(i))
      end if

      if (distrib_flag(i) /= 'O') then

        ! Probably outside grid range, so set value to that of the soft
        ! clipping.
        distrib(i) = 0.0

      end if

      func2d=0.0

    end do each_point

  end function func2d


    subroutine grid2d(a_par, dat, colnam, correlated, &
      var_par, start, end, soft_clip, &
      data, axdata, best_value, best_par, best_distrib, rnpts)

      use define_star

      ! Performs a 2d grid search.

      ! The model parameters.
      real, dimension(:) :: a_par
      ! The data points.
      type(a_star), dimension(:), intent(in) :: dat
      ! The model and colours being used.
      character(len=*), dimension(2), intent(inout) :: colnam
      ! The correlation flag for s/r likelihood.
      integer, intent(in) :: correlated
      ! The numbers of the parameters to be searched.
      integer, dimension(2), intent(in) :: var_par
      ! The ranges of parameters to be searched.
      real, dimension(2), intent(in) :: start, end
      real, intent(in) :: soft_clip
      ! The output grid.
      real, dimension(:,:), intent(out) :: data, axdata
      ! And an optional fraction of the stars on the grid.
      real, dimension(:,:), intent(out), optional :: rnpts

      real, intent(out) :: best_value
      logical :: first_value
      real, intent(out), dimension(:) :: best_par
      real, intent(out), dimension(:) :: best_distrib

      ! Locals.
      character, allocatable, dimension(:) :: distrib_flag
      real, allocatable, dimension(:) :: distrib
      integer :: i1, i2, i3
      real, dimension(size(data,1), size(data,2), size(dat,1)) :: distrib_full
      real :: dummy, work, low_tau
      integer, dimension(2) :: loc

      allocate(distrib_flag(size(dat,1)), distrib(size(dat,1)))

      ! Set up the values of best_par to be those of a_par.  We'll
      ! the correct the ones we fit.
      best_par=a_par

      first_value=.true.
      do i1=1, size(data,1)
        if (size(data,1) == 1) then
          a_par(var_par(1))=start(1)
        else
          a_par(var_par(1))=start(1)+(end(1)-start(1))*real(i1-1) &
          /real(size(data,1)-1)
        end if
        axdata(i1,1)=a_par(var_par(1))
        do i2=1, size(data,2)
          if (size(data,2) == 1) then 
            a_par(var_par(2))=start(2)
          else
            a_par(var_par(2))=start(2)+(end(2)-start(2))*real(i2-1) &
            /real(size(data,2)-1)
          end if
          axdata(i2,2)=a_par(var_par(2))
          if (present(rnpts)) then
            dummy=func2d(a_par, dat, colnam, correlated, &
                 distrib_full(i1,i2,:), distrib_flag, rnpts(i1, i2))
          else 
            dummy=func2d(a_par, dat, colnam, correlated, &
                 distrib_full(i1,i2,:), distrib_flag)
          end if
        end do
      end do

      ! Find the highest probability in the whole fit.
      low_tau=-2.0*log(maxval(distrib_full))
      print*, 'Lowest tau-squared for any point is ', low_tau
      if (soft_clip > 0.0) then
        work=exp(-0.5*(soft_clip+low_tau))
      else
        work=0.0
      end if
      ! Soft clip the probabilities.
      distrib_full=distrib_full+work

      data=0.0
      high_tau=-2.0*log(2.0*tiny(1.0))*size(dat,1)
      do i1=1, size(data,1)
        do i2=1, size(data,2)
          ! Construct the tau**2 for this grid point by adding
          ! the tau^2 of each point.
          each_point: do i3=1, size(dat,1)
            ! Protect against infinities.
            if (distrib_full(i1,i2,i3) > 2.0*tiny(1.0)) then
              data(i1,i2)=data(i1,i2)-2.0*log(distrib_full(i1,i2,i3))
            else
              ! At least one data point is returning a probablilty so low
              ! we can't cope, so set the entire grid point to a very
              ! high tau^2.
              data(i1,i2)=high_tau
              exit each_point
            end if
          end do each_point
        end do
      end do

      ! Find the best fit.
      loc=minloc(data)
      best_distrib=-2.0*log(distrib_full(loc(1),loc(2),:))
      best_par(var_par(1))=axdata(loc(1),1)
      best_par(var_par(2))=axdata(loc(2),2)
      best_value=sum(best_distrib)

      deallocate(distrib_flag, distrib)

    end subroutine grid2d

    subroutine tau_sort(tau_data, pix_prob)

      real, intent(inout), dimension(:)  :: tau_data
      real, optional, intent(inout), dimension(:)  :: pix_prob

      integer :: k, l, m
      real :: safe, safe_pix

      outer: do k=2, size(tau_data)
        safe=tau_data(k)
        if (present(pix_prob)) safe_pix=pix_prob(k)
        do l=1,k-1
          if (safe < tau_data(l)) then
            do m=k, l+1, -1
              tau_data(m)=tau_data(m-1)
              if (present(pix_prob)) pix_prob(m)=pix_prob(m-1)
            end do
            tau_data(l)=safe
            if (present(pix_prob)) pix_prob(l)=safe_pix
            cycle outer
          endif
        end do
      end do outer

    end subroutine tau_sort


    end module cmdfit_subs
