program monte

  use ark_file_io
  use cmdfit_subs
  use colteff_subs

  implicit none

  integer :: iflag, colflag, modflag, ipos
  integer :: iage, nage
  real :: age, age_low, age_high
  integer, dimension(2) :: naxis
  real, allocatable, dimension(:,:) :: data, axdata
  real, allocatable, dimension(:) :: grad
  character(len=50) :: ofname
  character(len=8) :: age_name
  integer :: i1, i2
  real :: col_start, col_end, mag_start, mag_end, low_mass, high_mass
  integer :: useful_top, useful_bottom
  ! The image, and the associated values in axis1 (colour) and axis2 (mag).
  type a_grader
    real, dimension(2) :: low, high
    integer :: npts
  end type a_grader
  type(a_grader), allocatable, dimension(:) :: grader

  integer :: i, imag, icol, istar
  real :: col, mag, p_mass
  character(len=50) :: isoflag
  ! The number of stars in the simulation.
  integer :: nstar
  ! The number of stars in the simple isochrone.
  integer, parameter :: niso=10000
  ! The binary fraction.
  real :: binary_fraction
  logical :: global_norm=.false.
  ! More edges.
  real :: mag_min, mag_max, col_min, col_max

  real :: red

  !call setbug()

  print*, 'What reddening do you require?'
  read(*,*) red

  ! This used to be hard coded -1.0 to 4.0.
  print*, 'What range of colours do you want (blue, red)?'
  read(*,*) col_start, col_end

  ! This used to be hard coded 0 to 15
  print*, 'What range of absolute magnitudes to you want to cover (bright, faint)?'
  read(*,*) mag_end, mag_start

  ! This used to be hard coded 0.04 3.0
  print*, 'What range of masses do you want (low, high)?'
  read(*,*) low_mass, high_mass
  
  print*, 'The following models are available.'
  do i=1, n_models
    if (len_trim(mod_name(i)) > 0) print*, i, trim(mod_name(i))
  end do
  print*, 'What model number do you want?'
  read(*,*) modflag

  print*, 'The available CMDs are '
  do i=1, n_names
    if (len_trim(mag_name(i)) > 0) & 
    print*, i, trim(mag_name(i)), ' vs ', trim(col_name(i)), &
    '   ('//trim(pht_comm(i))//')'
  end do
  print*, 'What colour number do you want?'
  read(*,*) colflag

  print*, 'What binary fraction do you want?'
  read(*,*) binary_fraction

  print*, 'What range of ages do you want?'
  read(*,*) age_low, age_high
  print*, 'And how many ages (remember n+1)?'
  read(*,*) nage


  do iage=1, nage

    if (nage == 1) then
      age = age_low
    else
      age = age_low + real(iage-1)*(age_high-age_low)/real(nage-1)
    end if

    print*, 'Doing ', age, 'Myr.'

    naxis(1)=nint(200.0*(col_end-col_start))+1
    naxis(2)=nint(200.0*(mag_start-mag_end))+1

    print*, 'Grid will be ', naxis

    ! Now calculate the number of stars required.  Used to be fixed at
    ! 1000000
    nstar=nint( 1000000.0/(15.0*200.0) )*real(naxis(2))
    print*, 'With ', nstar, ' simulated objects.'

    allocate(data(naxis(1),naxis(2)), axdata(maxval(naxis),2), &
    grad(naxis(1)), grader(naxis(1)))

    do i=1, naxis(1)
      axdata(i,1)=col_start+(col_end-col_start)*real(i-1)/real(naxis(1)-1)
    end do
    do i=1, naxis(2)
      axdata(i,2)=mag_start+(mag_end-mag_start)*real(i-1)/real(naxis(2)-1)
    end do

    ! Find the maximum available magnitude and colour.
    mag_max=mag_end   - 0.5*(mag_end-mag_start)/real(naxis(2)-1)
    mag_min=mag_start + 0.5*(mag_end-mag_start)/real(naxis(2)-1)
    col_max=col_end   + 0.5*(col_end-col_start)/real(naxis(1)-1)
    col_min=col_start - 0.5*(col_end-col_start)/real(naxis(1)-1)

    ! Write out a simple isochrone file.
    call iso_image_name(modflag, colflag, age, ofname)
    open (unit=12, file=trim(ofname)//'.iso')
    write(12,*) 'Colour, magnitude, mass, flag.'
    write(12,*)
    write(12,*)

    data=0.0
    grader%npts=0

    ! We need to calculate the grader array, so we create an array of
    ! single stars whatever the binary fraction is.
    each_single_star: do istar=1, nstar

      ! print*, istar

      call make_star(age, colflag, modflag, low_mass, high_mass, -0.1, &
      1, mag, col, p_mass, isoflag)
      if (istar <= niso) write(12,*) col, mag, p_mass, trim(isoflag)
      if (trim(isoflag) /= 'OK') cycle each_single_star
      ! If you change this make use you change the one below.
      !call make_star_ubv(age, modflag, low_mass, high_mass, -0.1, &
      !1, mag, col)

      ! Apply the reddening.
      call reddening(colflag, 0.0, red, col, mag)

      ! First throw out the stars outside the image.
      if (mag < mag_end)   cycle each_single_star
      if (mag > mag_start) cycle each_single_star
      if (col > col_end)   cycle each_single_star
      if (col < col_start) cycle each_single_star
      ! Find the pixel number.
      imag=minloc(abs(axdata(1:naxis(2),2)-mag),1)
      icol=minloc(abs(axdata(1:naxis(1),1)-col),1)  

      ! Put the single star into the image.
      data(icol,imag)=data(icol,imag)+1.0

      ! To find the gradient of the single star sequence we use the
      ! most extreme colour value points in column of pixels at a given
      ! colour.  So, is this point more extreme than any we've go so far?
      ! If it is, store it.
      if (grader(icol)%npts==0 .or. col<grader(icol)%low(1)) then
        grader(icol)%low(1)  = col
        grader(icol)%low(2)  = mag
        if (grader(icol)%npts /= 0) grader(icol)%npts=grader(icol)%npts+1
      end if
      if (grader(icol)%npts==0 .or. col>grader(icol)%high(1)) then
        grader(icol)%high(1) = col
        grader(icol)%high(2) = mag
        grader(icol)%npts=grader(icol)%npts+1
      end if

    end do each_single_star
    print*, 'Done singles.'
    close(12)

    ! Now calculate the gradient array.
    where (grader%npts > 1) 
      grad = (grader%high(2)-grader%low(2))/(grader%high(1)-grader%low(1))
    elsewhere
      grad = 0.0
    endwhere

    ! Now reduce this by the binary fraction.
    data=data*(1.0-binary_fraction)

    each_binary_star: do istar=1, nint(real(nstar)*binary_fraction)

      call make_star(age, colflag, modflag, low_mass, high_mass, -0.1, &
      2, mag, col, p_mass, isoflag)
      ! If you change this make use you change the one above.
      !call make_star_ubv(age, modflag, low_mass, high_mass, -0.1, &
      !2, mag, col)

      ! Apply the reddening.
      call reddening(colflag, 0.0, red, col, mag)

      if (trim(isoflag) /= 'OK') cycle each_binary_star
      if (mag < mag_end)   cycle each_binary_star
      if (mag > mag_start) cycle each_binary_star
      if (col > col_end)   cycle each_binary_star
      if (col < col_start) cycle each_binary_star

      ! Find the pixel number.
      imag=minloc(abs(axdata(1:naxis(2),2)-mag),1)
      icol=minloc(abs(axdata(1:naxis(1),1)-col),1)  

      ! Put the binary star into the image.
      data(icol,imag)=data(icol,imag)+1.0

    end do each_binary_star
    print*, 'Done binaries.'

    ! Remove colours outside the range of the gradient, otherwise where you
    ! divide by the gradient in the fitting routine causes a crash.
    do i=1, naxis(1)
      if (grader(i)%npts == 0) data(i,:)=0.0 
    end do
    deallocate(grader)

    ! Normalise so the mean is one in every magnitude bin. 
    if (mag_name(colflag)(2:2) == '-') then
      ! I.e. if its a colour-colour diagram.
      !  global_norm=.true.
      do i1=1, naxis(1)
        if (sum(data(i1,1:naxis(2))) > 0.0) then
          data(i1,1:naxis(2))=data(i1,1:naxis(2))/maxval(data(i1,1:naxis(2))) 
        end if
      end do
    else
      do i2=1, naxis(2)
        if (sum(data(1:naxis(1),i2)) > 0.0) then
          data(1:naxis(1),i2)=data(1:naxis(1),i2)/sum(data(1:naxis(1),i2)) 
        end if
      end do
    end if

    ! Subset the data.
    useful_bottom=0
    useful_top=naxis(2)+1
    do i2=1, naxis(2)
      if (sum(data(1:naxis(1),i2)) > 0.0) then
        useful_bottom=i2
        useful_top=min(useful_top,i2)
      end if
    end do
    naxis(2)=useful_bottom-useful_top+1
    data(1:naxis(1), 1:naxis(2))=data(1:naxis(1), useful_top:useful_bottom)
    axdata(1:naxis(2),2)=axdata(useful_top:useful_bottom,2)

    
    if (global_norm) then
      data=data/sum(data)
    else
      ! Now make it a density per magnitude in colour.
      data=data/(axdata(2,1)-axdata(1,1))
    end if

    ! First write out the image file.
    call put_header_i('MODFLAG', modflag, 'Whose isochrones' , 1)
    call put_header_i('COLFLAG', colflag, 'Which colour' , 1)
    call put_header_r('BINFRAC', binary_fraction, 'Binary star fraction' , 1)
    call nxtark_out(ofname)
    iflag=makark(naxis, data, axdata)

    ! Now write out the gradient file, unless is a colour-colour diagram.
    if (mag_name(colflag)(2:2) /= '-') then
      ipos=index(ofname, '.fit')
      ofname(ipos+1:ipos+3)='grd'
      call nxtark_out(ofname)
      iflag=makark(naxis(1), grad, axdata(:,1))
    end if

    deallocate(data, axdata, grad)

  end do

end program monte
